Attribute VB_Name = "ODBC32"
'------------------------------------------------------------------
' Name : ODBC32
'
' Purpose : Purpose an interface to access to Database
'
' Method:
'   1) BeginTran          Begin a transaction
'   2) CommitTran         Commit a transaction
'   3) RollBackTran       Cancel a transaction
'   4) ErrorGet           Retrieves the error of the environment,
'                         database and statement (Private)
'   5) ODBCConnectBuild   Build the Connection string
'   6) ODBCData           Retrieve one column of the current row
'   7) ODBCDataClose      Close the connection with the database
'   8) ODBCDataOpen       Open the connection with the database
'   9) ODBCInit           Initialize the connection with a database
'  10) ODBCMemoGet        Retrieve a value in the Text Format (>255 char)
'  11) QuoteParam         Verify if the string has quote and modifiy the
'                         String if necessary
'  12) SQLSubmit          Run a request to the server
'  13) SQLSubmitServer    Run a request to the server with server
'                         cursor (For compatibility, must use SQLSubmit now)
'  14) SQLSubmitError     Send a request to the server to write an
'                         error message into the database
'  15) SQLFreeStatement   Call SQLFreeStmt if necessary
'  16) SQLFetch           Call SQLFetch (ODBC) if necessary
'
' review : Jan/28/1999 by AD
'------------------------------------------------------------------
Option Explicit

Global Const Environment = "Win 32"
Global Const TRY_COUNT As Integer = -2

Global gs_TraceUser As String
Global gs_ErrorMesssage() As String
Global gs_NativeError() As Long
Global gi_ErrorCount As Integer
Global gb_NoMessage As Boolean

Private ms_UserODBC As String
Private ms_Directory As String

'False by default
Public mb_CacheODBC As Boolean
Public mb_UseCache As Boolean

Private mb_FirstRequest As Boolean
Private mo_BinaryFile As BinaryFile
Private ms_Data() As String
Private mo_StoredProcedureList As New Collection

Private mb_LockedStatement As Boolean
Private mb_FirstTest As Boolean
Private mb_NCS_Method As Boolean

Private Const CACHE_DIRECTORY = ""
Private Const CACHE_EXTENSION = "lch"

'Declare Type for the column description
Private Type TypeColumnDescription
    ColumnName As String * 20
    DataType As Integer
    SizeColName As Integer
    SizeColumn As Long
    Scale As Integer
    Nullable As Integer
End Type

Dim met_DescColumns() As TypeColumnDescription

Private Type TypeStatement
    Statement As Long
    Connection As Integer
End Type

Private Type TypeConnection
    Connection As Integer
    Environment As Long
    Database As Long
End Type

Dim met_Statement() As TypeStatement
Dim met_Connection() As TypeConnection
Dim mi_ConnectionNumber As Integer
Dim mi_StatementNumber As Integer
Dim ml_CurrentEnvironment As Long
Dim ml_CurrentDatabase As Long
Dim mi_CurrentConnection As Integer
Dim ms_ConnectionString As String

'****** SQL Constants ***************
Public Const MAX_DATA_BUFFER As Integer = 255

'
'|========================================================================|
'| ODBC Module Core Definitions                                           |
'|========================================================================|
'
'  ODBC Core API's Definitions -- 32 bit versions
'
Private Declare Function SQLAllocConnect Lib "odbc32.dll" (ByVal henv&, phdbc&) As Integer
Private Declare Function SQLAllocEnv Lib "odbc32.dll" (phenv&) As Integer
Private Declare Function SQLAllocStmt Lib "odbc32.dll" (ByVal hdbc&, phstmt&) As Integer
        Declare Function SQLBindCol Lib "odbc32.dll" (ByVal hstmt&, ByVal icol%, ByVal fCType%, rgbValue As Any, ByVal cbValueMax&, pcbValue&) As Integer

        Declare Function SQLDescribeCol Lib "odbc32.dll" (ByVal hstmt&, ByVal icol%, ByVal SzColName$, ByVal cbColNameMax%, pcbColName%, pfSqlType%, pcbColDef&, pibScale%, pfNullable%) As Integer
Private Declare Function SQLDisconnect Lib "odbc32.dll" (ByVal hdbc&) As Integer
Private Declare Function SQLError Lib "odbc32.dll" (ByVal henv&, ByVal hdbc&, ByVal hstmt&, ByVal szSqlState$, pfNativeError&, ByVal szErrorMsg$, ByVal cbErrorMsgMax%, pcbErrorMsg%) As Integer
Private Declare Function SQLExecDirect Lib "odbc32.dll" (ByVal hstmt&, ByVal szSqlStr$, ByVal cbSqlStr&) As Integer
Private Declare Function SQLFetch1 Lib "odbc32.dll" Alias "SQLFetch" (ByVal hstmt&) As Integer
Private Declare Function SQLFreeConnect Lib "odbc32.dll" (ByVal hdbc&) As Integer
Private Declare Function SQLFreeEnv Lib "odbc32.dll" (ByVal henv&) As Integer
Private Declare Function SQLFreeStmt Lib "odbc32.dll" (ByVal hstmt&, ByVal fOption%) As Integer
        Declare Function SQLNumResultCols Lib "odbc32.dll" (ByVal hstmt&, pccol%) As Integer
Private Declare Function SQLRowCount1 Lib "odbc32.dll" Alias "SQLRowCount" (ByVal hstmt&, pcrow&) As Integer

'
'|========================================================================|
'| ODBC Module Extended Definitions                                       |
'|========================================================================|
''  Level 1 Prototypes
'

Private Declare Function SQLDriverConnect Lib "odbc32.dll" (ByVal hdbc&, ByVal hwnd As Long, ByVal szCSIn$, ByVal cbCSIn%, ByVal szCSOut$, ByVal cbCSMax%, cbCSOut%, ByVal fDrvrComp%) As Integer

Private Declare Function SQLGetDataString Lib "odbc32.dll" Alias "SQLGetData" (ByVal hstmt&, ByVal icol%, ByVal fCType%, ByVal rgbValue As String, ByVal cbValueMax&, pcbValue&) As Integer

Private Declare Function SQLGetDataOther Lib "odbc32.dll" Alias "SQLGetData" (ByVal hstmt&, ByVal icol%, ByVal fCType%, rgbValue As Any, ByVal cbValueMax&, pcbValue&) As Integer

Private Declare Function SQLSetStmtOption Lib "odbc32.dll" (ByVal hstmt&, ByVal fOption%, ByVal vParam&) As Integer

'  Level 2 Prototypes
'
        Declare Function SQLExtendedFetch Lib "odbc32.dll" (ByVal hstmt&, ByVal fFetchType%, ByVal iRow&, pcrow&, rgfRowStatus%) As Integer

'
'|========================================================================|
'| Install ODBC Module Definitions                                        |
'|========================================================================|
'

        Declare Function SQLConfigDataSource Lib "ODBCCP32.DLL" (ByVal hwndParent As Long, ByVal fRequest As Long, ByVal lpszDriver As String, ByVal lpszAttributes As String) As Long


'
'  32 Bit declares
'  ODBC Constants/Types
'
'  RETCODEs
'
Global Const SQL_ERROR As Long = -1
Global Const SQL_SUCCESS As Long = 0
Global Const SQL_SUCCESS_WITH_INFO As Long = 1

'  SQLFreeStmt defines
'
Global Const SQL_CLOSE As Long = 0
Global Const SQL_DROP As Long = 1

'  Standard SQL datatypes, using ANSI type numbering
'
Global Const SQL_CHAR As Long = 1
Global Const SQL_NUMERIC As Long = 2
Global Const SQL_DECIMAL As Long = 3
Global Const SQL_INTEGER As Long = 4
Global Const SQL_SMALLINT As Long = 5
Global Const SQL_FLOAT As Long = 6
Global Const SQL_REAL As Long = 7
Global Const SQL_DOUBLE As Long = 8
Global Const SQL_VARCHAR As Long = 12

Global Const SQL_TYPE_MIN As Long = 1
Global Const SQL_TYPE_NULL As Long = 0
Global Const SQL_TYPE_MAX As Long = 12

'  C datatype to SQL datatype mapping    SQL types
'
Global Const SQL_VB_CHAR    As Integer = SQL_CHAR           '  CHAR, VARCHAR, DECIMAL, NUMERIC
Global Const SQL_VB_LONG    As Integer = SQL_INTEGER        '  INTEGER
Global Const SQL_VB_INTEGER As Integer = SQL_SMALLINT       '  SMALLINT
Global Const SQL_VB_REAL    As Integer = SQL_REAL           '  REAL
Global Const SQL_VB_DOUBLE  As Integer = SQL_DOUBLE         '  FLOAT, DOUBLE

'  Special length values
'
Global Const SQL_NULL_DATA As Long = -1

'  SQLColAttributes defines
'
Global Const SQL_COLUMN_COUNT As Long = 0
Global Const SQL_COLUMN_NAME As Long = 1
Global Const SQL_COLUMN_TYPE As Long = 2
Global Const SQL_COLUMN_LENGTH As Long = 3
Global Const SQL_COLUMN_PRECISION As Long = 4
Global Const SQL_COLUMN_SCALE As Long = 5
Global Const SQL_COLUMN_DISPLAY_SIZE As Long = 6
Global Const SQL_COLUMN_NULLABLE As Long = 7
Global Const SQL_COLUMN_UNSIGNED As Long = 8
Global Const SQL_COLUMN_MONEY As Long = 9
Global Const SQL_COLUMN_UPDATABLE As Long = 10
Global Const SQL_COLUMN_AUTO_INCREMENT As Long = 11
Global Const SQL_COLUMN_CASE_SENSITIVE As Long = 12
Global Const SQL_COLUMN_SEARCHABLE As Long = 13
Global Const SQL_COLUMN_TYPE_NAME As Long = 14
Global Const SQL_COLUMN_TABLE_NAME As Long = 15
Global Const SQL_COLUMN_OWNER_NAME As Long = 16
Global Const SQL_COLUMN_QUALIFIER_NAME As Long = 17
Global Const SQL_COLUMN_LABEL As Long = 18

'  SQLError defines
'
Global Const SQL_NULL_HENV As Long = 0
Global Const SQL_NULL_HDBC As Long = 0
Global Const SQL_NULL_HSTMT As Long = 0
Global Const SQL_NULL_STATEMENT As Integer = 0

'|========================================================================|
'| ODBC Global Extended Definitions                                       |
'|========================================================================|

' Options for SQLDriverConnect
'
Global Const SQL_DRIVER_NOPROMPT As Long = 0

' Options for SQLGetStmtOption/SQLSetStmtOption
'
Global Const SQL_QUERY_TIMEOUT As Long = 0
Global Const SQL_MAX_ROWS As Long = 1
Global Const SQL_NOSCAN As Long = 2
Global Const SQL_MAX_LENGTH As Long = 3
Global Const SQL_ASYNC_ENABLE As Long = 4
Global Const SQL_BIND_TYPE As Long = 5
Global Const SQL_CURSOR_TYPE As Long = 6
Global Const SQL_CONCURRENCY As Long = 7
Global Const SQL_KEYSET_SIZE As Long = 8
Global Const SQL_ROWSET_SIZE As Long = 9
Global Const SQL_SIMULATE_CURSOR As Long = 10
Global Const SQL_RETRIEVE_DATA As Long = 11
Global Const SQL_USE_BOOKMARKS As Long = 12
Global Const SQL_GET_BOOKMARK As Long = 13
Global Const SQL_ROW_NUMBER As Long = 14

Global Const SQL_STMT_OPT_MAX As Long = SQL_ROW_NUMBER

' Statement option values & defaults
'
Global Const SQL_BIND_BY_COLUMN As Long = 0

Global Const SQL_CONCUR_READ_ONLY As Long = 1
Global Const SQL_CONCUR_LOCK As Long = 2
Global Const SQL_CONCUR_ROWVER As Long = 3
Global Const SQL_CONCUR_VALUES As Long = 4

' Level 2 Functions
'
' SQLExtendedFetch "fFetchType" values
'
Global Const SQL_FETCH_NEXT As Long = 1
Global Const SQL_FETCH_FIRST As Long = 2
Global Const SQL_FETCH_LAST As Long = 3
Global Const SQL_FETCH_PRIOR As Long = 4
Global Const SQL_FETCH_ABSOLUTE As Long = 5
Global Const SQL_FETCH_RELATIVE As Long = 6
Global Const SQL_FETCH_BOOKMARK As Long = 8


' SQLExtendedFetch "rgfRowStatus" element values
'
Global Const SQL_ROW_SUCCESS As Long = 0
Global Const SQL_ROW_DELETED As Long = 1
Global Const SQL_ROW_UPDATED As Long = 2
Global Const SQL_ROW_NOROW As Long = 3

' Deprecated global constants
'
Global Const SQL_CURSOR_FORWARD_ONLY = 0
Global Const SQL_CURSOR_KEYSET_DRIVEN = 1
Global Const SQL_CURSOR_DYNAMIC = 2
Global Const SQL_CURSOR_STATIC = 3

Global Const FETCH_64K = 0
Global Const SINGLE_FETCH = 1
Global Const MULTIROW_FETCH = 2

Sub ChangeUser(ByRef ls_User As String)
'------------------------------------------------------------------
' Name : ChangeUser
'
' Purpose : Changue the ODBC User
'
' Parameters :
'       ls_User        ODBC User to change
'
' Return : Nothing
'
' review : Jan/28/1999 by AD
'------------------------------------------------------------------

    ms_UserODBC = ls_User
    
End Sub

Sub CacheInit(ls_AppDirectory As String)
'------------------------------------------------------------------
' Name : CacheInit
'
' Purpose : Initialize the cache
'
' Parameters :
'       ls_AppDirectory     Directory of the application
'
' Return : Nothing
'
' review : Jan/28/1999 by AD
'------------------------------------------------------------------
Dim i As Integer

    'Free the collection
    For i = mo_StoredProcedureList.Count To 1 Step -1
        mo_StoredProcedureList.Remove i
    Next
    
    mb_CacheODBC = OK
    mb_FirstRequest = OK
    
    ms_Directory = ls_AppDirectory
    
    'we create a new Binary file instance
    Set mo_BinaryFile = New BinaryFile
    mo_BinaryFile.OverWrite = OK
    mo_BinaryFile.ForceClose = OK
    
End Sub

Function SQLFetch(ByVal ll_Statement As Long) As Integer
'------------------------------------------------------------------
' Name : SQLFetch
'
' Purpose : Call SQLFetch from ODBC but verify if the statement is
'           correct. Read a new line if we use cache
'
' Parameters :
'       ll_Statement        Statement identifier
'
' Return :
'       Status of the request
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim lb_Result As BinaryFile_Error

    If mb_UseCache = OK Then
        'We read a new line
        lb_Result = mo_BinaryFile.ReadLine(ms_Data)
        If lb_Result <> BF_OK Then
            SQLFetch = SQL_ERROR
        Else
            SQLFetch = SQL_SUCCESS
        End If
    Else
        If ll_Statement <> 0 Then
            SQLFetch = SQLFetch1(ll_Statement)
        Else
            SQLFetch = SQL_ERROR
        End If
    End If
    
End Function

Function SQLRowCount(ByVal ll_Statement As Long, ByRef ll_rowNumber As Long) As Integer
'------------------------------------------------------------------
' Name : SQLRowCount
'
' Purpose : Call SQLRowCount from ODBC but verify if the statement is
'           correct. Return SQL_ERROR if it's cache
'
' Parameters :
'       ll_Statement        Statement identifier
'       ll_rowNumber        Return the number of row
'
' Return :
'       Status of the request
'
' review : Nov/23/1999 by AD
'------------------------------------------------------------------
Dim ll_rowNumber1 As Long
    
    If mb_UseCache = OK Then
        SQLRowCount = SQL_ERROR
    Else
        If ll_Statement <> 0 Then
            SQLRowCount = SQLRowCount1(ll_Statement, ll_rowNumber1)
            ll_rowNumber = ll_rowNumber1
        Else
            SQLRowCount = SQL_ERROR
        End If
    End If
    
End Function

Private Sub ErrorGet(ByVal ll_Environment, ByVal ll_Database, ByVal ll_Statement)
'------------------------------------------------------------------
' Name : ErrorGet
'
' Purpose : Retrieve the Environment, Database and Statemnt errors
'
' Parameters :
'       ll_Environment      Environment identifier
'       ll_Database         Database identifier
'       ll_Statement        Statement identifier
'
' review : Oct/17/1999 by AD
'------------------------------------------------------------------
Dim li_Loop As Integer
Dim li_Count As Integer
Dim li_Status As Integer
Dim ls_SQLState As String * 16
Dim ls_ErrMsg As String * MAX_DATA_BUFFER
Dim li_OutLen As Integer
Dim ll_Native As Long
Dim ll_Environment2 As Long
Dim ll_Database2 As Long
Dim ll_Statement2 As Long
   
    'Initialize different variables
    ls_SQLState = String$(16, 0)
    ls_ErrMsg = String$(MAX_DATA_BUFFER, 0)
    
    'Initialise the error arrays
    ReDim gs_ErrorMesssage(0)
    ReDim gs_NativeError(0)
    gi_ErrorCount = 0
    
    li_Count = 0
    ' Loop through all 3 error state types.
    For li_Loop = 1 To 3
        'For each loop initialize the variables
        Select Case li_Loop
            Case 1
                ll_Environment2 = SQL_NULL_HENV
                ll_Database2 = SQL_NULL_HDBC
                ll_Statement2 = ll_Statement
            Case 2
                ll_Environment2 = SQL_NULL_HENV
                ll_Database2 = ll_Database
                ll_Statement2 = SQL_NULL_HSTMT
            Case 3
                ll_Environment2 = ll_Environment
                ll_Database2 = SQL_NULL_HDBC
                ll_Statement2 = SQL_NULL_HSTMT
        End Select
        Do
            'Execute the request to the server
            li_Status = SQLError(ll_Environment2, ll_Database2, ll_Statement2, ls_SQLState, ll_Native, ls_ErrMsg, MAX_DATA_BUFFER, li_OutLen)
            'If OK, we read the message error
            If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
                li_Count = li_Count + 1
                ReDim Preserve gs_ErrorMesssage(li_Count)
                ReDim Preserve gs_NativeError(li_Count)
                If li_OutLen = 0 Then
                    ReDim Preserve gs_ErrorMesssage(li_Count)
                    ReDim Preserve gs_NativeError(li_Count)
                    gs_ErrorMesssage(li_Count) = " Error -- No error information available"
                Else
                    gs_ErrorMesssage(li_Count) = Left$(ls_ErrMsg, li_OutLen)
                    gs_NativeError(li_Count) = ll_Native
                End If
            End If
        Loop Until li_Status <> SQL_SUCCESS
        
    Next li_Loop
    
    gi_ErrorCount = li_Count

End Sub

Function ODBCConnectBuild(ByVal ls_DSN As String, ByVal ls_DatabaseName As String, ByVal ls_AppIdentifier As String, ByVal ls_Login As String, ByVal ls_Password As String, Optional ls_OtherParameters As String) As String
'------------------------------------------------------------------
' Name : ODBCConnectBuild
'
' Purpose : Construct the Connection string
'
' Parameters :
'       ls_DSN              Indicates the ODBC DSN Name
'       ls_DatabaseName     Name of the database to access
'       ls_AppIdentifier    String that will be indicates in SQL Server
'       ls_Login            Login to use for identification
'       ls_Password         Password to use for identification
'
' review : Sep/17/1999 by AD
'------------------------------------------------------------------
Dim ls_ConnectionString As String

 ' "DRIVER={SQL Server};SERVER=MyServer;"
   '      "UID=sa;PWD=MyPassWord;DATABASE=pubs;";

    ls_ConnectionString = "DRIVER={SQL Server};SERVER=" & gs_ServerIP
    ls_ConnectionString = ls_ConnectionString & ";UID=" & ls_Login
    ls_ConnectionString = ls_ConnectionString & ";PWD=" & ls_Password
    ls_ConnectionString = ls_ConnectionString & ";Database=" & gs_DBName
   ' If ls_DatabaseName <> "" Then ls_ConnectionString = ls_ConnectionString & ";DATABASE=" & ls_DatabaseName
    ls_ConnectionString = ls_ConnectionString & ";APP=" & ls_AppIdentifier
'    If Not (IsMissing(ls_OtherParameters) = True) Then
'        ls_ConnectionString = ls_ConnectionString & ";" & ls_OtherParameters
'    End If
    ms_UserODBC = ls_Login
    
    ODBCConnectBuild = ls_ConnectionString
End Function

Function ODBCInit(ByRef ll_Environment As Long, ByRef ll_Database As Long) As Boolean
'------------------------------------------------------------------
' Name : ODBCInit
'
' Purpose : Initialize the connection to the database
'
' Parameters :
'       ll_Environment      Environment identifier
'       ll_Database         Database identifier
'
' review : Sep/17/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ll_Environment1 As Long
Dim ll_Database1 As Long

    On Error GoTo suite
    
    'Allocation of the environment
    If SQLAllocEnv(ll_Environment1) = SQL_ERROR Then
        If gb_NoMessage = False Then
            MouseOn
            MsgBox "Can't Allocate Memory For Environment Handle"
            MouseOff
        End If
        ODBCInit = KO
    Else
        ' This Call establishes a memory area where we can
        ' put login information that will be used to login
        ' to our data source. The area for the login
        ' information is contained in the hdbc variable.
        If SQLAllocConnect(ll_Environment1, ll_Database1) = SQL_ERROR Then
            If gb_NoMessage = False Then
                MouseOn
                MsgBox "Can't Allocate Memory For Connection Handle"
                MouseOff
            End If
            ODBCInit = KO
            ' Free the Environment
            li_Status = SQLFreeEnv(ll_Environment)
            If li_Status = SQL_ERROR Then
                If gb_NoMessage = False Then
                    MouseOn
                    MsgBox "Can't Release Memory for Environment Handle"
                    MouseOff
                End If
            End If
        Else
            ODBCInit = OK
            ll_Environment = ll_Environment1
            ll_Database = ll_Database1
        End If
    End If
    Exit Function
    
suite:
    StdError
End Function

Function ODBCDataOpen(ByVal ll_Database As Long, ByVal ls_Connect As String) As Integer
'------------------------------------------------------------------
' Name : ODBCDataOpen
'
' Purpose : Open the connection to the database
'
' Parameters :
'       ll_Database         Database identifier
'       ls_Connect          Connection String
'
' review : Sep/17/1999 by AD
'------------------------------------------------------------------
Static si_TryCount As Integer
Dim ls_Message As String
Dim ls_Result As String
Dim li_size As Integer
Dim li_Status As Integer
Dim ls_Directory As String

    ls_Result = SPACE$(255)
  
    'Execute the request to the server
    li_Status = SQLDriverConnect(ll_Database, 0, ls_Connect, Len(ls_Connect), ls_Result, Len(ls_Result), li_size, SQL_DRIVER_NOPROMPT)
    If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
        ' Valid Logon
        ODBCDataOpen = OK
        
        If mb_CacheODBC = OK Then
            ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
            
            'Verify if the cache directory exist
            If Dir(ls_Directory, vbDirectory) = "" Then
                MkDir ls_Directory
            End If
        End If
    Else
        If gb_NoMessage = False Then
            'We try only 4 times
            si_TryCount = si_TryCount + 1
            If si_TryCount < 4 Then
                'it's not the last times
                ls_Message = "Can't connect to server :" & Chr(10) _
                    & " - wrong password or username," & Chr(10) _
                    & " - server not available." & Chr(10) _
                    & "  Try again !"
                    ODBCDataOpen = KO
            Else
                'it's the last times
                ls_Message = "You will be disconnect. Reboot your PC." & Chr(10) _
                    & "  If problem remains, contact your system administrator."
                ODBCDataOpen = TRY_COUNT
            End If
            MouseOn
            MsgBox ls_Message
            MouseOff
        End If
    End If

End Function

Function ODBCDataSmallInt(ByVal ll_Statement As Long, ByVal li_NbColumn As Integer) As String
'------------------------------------------------------------------
' Name : ODBCDataSMallInt
'
' Purpose : Retrieves one column of the current row in Integer format
'           (SmallInt for SQLServer)
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_NbColumn         column number to retrieve
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim li_Data As Integer
Dim ll_OutLen As Long

    If mb_UseCache = OK Then
        ODBCDataSmallInt = ODBCDataCache(li_NbColumn)
    Else
        If ll_Statement = 0 Then
            ODBCDataSmallInt = ""
            Exit Function
        End If
        
        li_Status = SQLGetDataOther(ll_Statement, li_NbColumn, SQL_SMALLINT, li_Data, 0, ll_OutLen)
        
        If li_Status <> SQL_ERROR And ll_OutLen > 0 Then
            ODBCDataSmallInt = Formatage2(CStr(li_Data))
        Else
            ODBCDataSmallInt = ""
        End If
    End If
    
End Function

Function ODBCDataInt(ByVal ll_Statement As Long, ByVal li_NbColumn As Integer) As String
'------------------------------------------------------------------
' Name : ODBCDataInt
'
' Purpose : Retrieves one column of the current row in Long format
'           (Integer for SQLServer)
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_NbColumn         column number to retrieve
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ll_Data As Long
Dim ll_OutLen As Long

    If mb_UseCache = OK Then
        ODBCDataInt = ODBCDataCache(li_NbColumn)
    Else
        If ll_Statement = 0 Then
            ODBCDataInt = ""
            Exit Function
        End If
        
        li_Status = SQLGetDataOther(ll_Statement, li_NbColumn, SQL_INTEGER, ll_Data, 0, ll_OutLen)
        
        If li_Status <> SQL_ERROR And ll_OutLen > 0 Then
            ODBCDataInt = Formatage2(CStr(CDec(ll_Data)))
        Else
            ODBCDataInt = ""
        End If
    End If
    
End Function

Function ODBCDataSingle(ByVal ll_Statement As Long, ByVal li_NbColumn As Integer) As String
'------------------------------------------------------------------
' Name : ODBCDataSingle
'
' Purpose : Retrieves one column of the current row in Single format
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_NbColumn         column number to retrieve
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ls_Data As Single
Dim ll_OutLen As Long

    If mb_UseCache = OK Then
        ODBCDataSingle = ODBCDataCache(li_NbColumn)
    Else
        If ll_Statement = 0 Then
            ODBCDataSingle = ""
            Exit Function
        End If
        li_Status = SQLGetDataOther(ll_Statement, li_NbColumn, SQL_REAL, ls_Data, 0, ll_OutLen)
        
        If li_Status <> SQL_ERROR And ll_OutLen > 0 Then
            ODBCDataSingle = Formatage2(CStr(CDec(ls_Data)))
        Else
            ODBCDataSingle = ""
        End If
    End If
    
End Function

Function ODBCData(ByVal ll_Statement As Long, ByVal li_NbColumn As Integer) As String
'------------------------------------------------------------------
' Name : ODBCData
'
' Purpose : Retrieves one column of the current row
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_NbColumn         column number to retrieve
'
' review : Oct/291999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ls_Data As String * MAX_DATA_BUFFER
Dim ll_OutLen As Long

    If mb_UseCache = OK Then
        ODBCData = ODBCDataCache(li_NbColumn)
    Else
        If li_NbColumn > UBound(met_DescColumns) Then
            If ll_Statement = 0 Then
                ODBCData = ""
                Exit Function
            End If
            li_Status = SQLGetDataString(ll_Statement, li_NbColumn, SQL_CHAR, ls_Data, MAX_DATA_BUFFER, ll_OutLen)
            If ll_OutLen < 1 Then
                ODBCData = ""
            Else
                ODBCData = Trim$(Left$(ls_Data, ll_OutLen))
            End If
        Else
            ' select the type of data to choice and count the type of different buffer
            Select Case met_DescColumns(li_NbColumn).DataType
                Case SQL_INTEGER ' VB Long
                    ODBCData = ODBCDataInt(ll_Statement, li_NbColumn)
                Case SQL_SMALLINT ' VB Integer
                    ODBCData = ODBCDataSmallInt(ll_Statement, li_NbColumn)
                Case SQL_FLOAT, SQL_DOUBLE, SQL_REAL ' VB Single
                    ODBCData = ODBCDataSingle(ll_Statement, li_NbColumn)
                Case Else
                    If ll_Statement = 0 Then
                        ODBCData = ""
                        Exit Function
                    End If
                    li_Status = SQLGetDataString(ll_Statement, li_NbColumn, SQL_CHAR, ls_Data, MAX_DATA_BUFFER, ll_OutLen)
                    If ll_OutLen < 1 Then
                        ODBCData = ""
                    Else
                        ODBCData = Trim$(Left$(ls_Data, ll_OutLen))
                    End If
            End Select
        End If
    End If
    
End Function

Private Function ODBCDataCache(ByVal li_NbColumn As Integer) As String
'------------------------------------------------------------------
' Name : ODBCDataCache
'
' Purpose : Retrieves one column of the current row in the cache
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_NbColumn         column number to retrieve
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------

    ODBCDataCache = ""
    If li_NbColumn <= UBound(ms_Data) Then
        ODBCDataCache = ms_Data(li_NbColumn - 1)
    End If
    
End Function

Function ODBCMemoGet(ByVal ll_Statement As Long, ByVal li_Column As Integer, Optional ab_Unicode As Boolean = False) As String
'------------------------------------------------------------------
' Name : ODBCMemoGet
'
' Purpose : Retrieve a value of Text Type (>255 char)
'
' Parameters :
'       ll_Statement        Statement identifier
'       li_Column           Column number to retrieve
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ls_Data As String * MAX_DATA_BUFFER
Dim ls_Value As String
Dim lb_Perform As Boolean
Dim ll_OutLen As Long

    On Error GoTo suite
    
    If mb_UseCache = OK Then
        ODBCMemoGet = ODBCDataCache(li_Column)
    Else
        If ll_Statement = 0 Then
            ODBCMemoGet = ""
            Exit Function
        End If
        
        ls_Data = ""
        
        lb_Perform = OK
        'Make a loop while the string is not finished
        Do While lb_Perform
            'Retrieves the next 255 chars
            li_Status = SQLGetDataString(ll_Statement, li_Column, 1, ls_Data, MAX_DATA_BUFFER, ll_OutLen)
            Select Case li_Status
            Case SQL_SUCCESS_WITH_INFO, SQL_SUCCESS
                'Adds these chars to the rest of the string
                If ll_OutLen >= MAX_DATA_BUFFER Or ll_OutLen < 0 Then
                    If ab_Unicode = True Then
                        ls_Value = ls_Value & Left$(ls_Data, MAX_DATA_BUFFER - 2)
                    Else
                        ls_Value = ls_Value & Left$(ls_Data, MAX_DATA_BUFFER - 1)
                    End If
                ElseIf ll_OutLen = SQL_NULL_DATA Then
                    ls_Value = ls_Value & ""
                Else
                    ls_Value = ls_Value & Left$(ls_Data, ll_OutLen)
                End If
                lb_Perform = IIf(li_Status = SQL_SUCCESS, 0, -1)
            Case Else
                lb_Perform = KO
            End Select
        Loop
        ODBCMemoGet = RTrim(ls_Value)
    End If

    Exit Function

suite:
    StdError
End Function

Sub ODBCDataClose(ByRef ll_Environment As Long, ByRef ll_Database As Long)
'------------------------------------------------------------------
' Name : ODBCDataClose
'
' Purpose : Close the connection to the database and delete the
'           Environment
'
' Parameters :
'       ll_Environment      Environment identifier
'       ls_Database         Database identifier
'
' review : Sep/17/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
   
    'Release binary file object
    Set mo_BinaryFile = Nothing
    
    If ll_Database <> 0 Then
    '   Logout of Data Source
        li_Status = SQLDisconnect(ll_Database)
        If li_Status = SQL_ERROR Then
            MouseOn
            MsgBox "Error Logging out of Data Source"
            MouseOff
        End If
    '   Free Connection Handle
        li_Status = SQLFreeConnect(ll_Database)
        If li_Status = SQL_ERROR Then
            MouseOn
            MsgBox "Error Freeing Connection Handle"
            MouseOff
        End If
        ll_Database = 0
    End If
    If ll_Environment <> 0 Then
    '   Free ODBC.DLL
        li_Status = SQLFreeEnv(ll_Environment)
        If li_Status = SQL_ERROR Then
            MouseOn
            MsgBox "Error Freeing Environment Handle"
            MouseOff
        End If
        ll_Environment = 0
    End If

End Sub

Public Function BeginTran(ByVal ll_Environment As Long, ByVal ll_Database As Long) As Boolean
'------------------------------------------------------------------
' Name : BeginTran
'
' Purpose : Begin a transaction
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim ls_req As String
Dim ll_Statement As Long
Dim li_Status As Integer

    BeginTran = KO
    
    On Error GoTo suite
    
    ls_req = "BEGIN TRANSACTION SIFYB"
    If SQLSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
        li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
        BeginTran = OK
    End If
    Exit Function
    
suite:
    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
    StdError
End Function

Public Function CommitTran(ByVal ll_Environment As Long, ByVal ll_Database As Long) As Boolean
'------------------------------------------------------------------
' Name : CommitTran
'
' Purpose : Commit a transaction
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim ls_req As String
Dim ll_Statement As Long
Dim li_Status As Integer
    
    CommitTran = KO
    
    On Error GoTo suite
    
    ls_req = "COMMIT TRANSACTION SIFYB"
    If SQLSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
        li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
        CommitTran = OK
    End If
    Exit Function
    
suite:
    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
    StdError
End Function

Public Function RollbackTran(ByVal ll_Environment As Long, ByVal ll_Database As Long) As Boolean
'------------------------------------------------------------------
' Name : RollBackTran
'
' Purpose : Cancel a transaction
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim ls_req As String
Dim ll_Statement As Long
Dim li_Status As Integer
    
    RollbackTran = KO
    
    On Error GoTo suite
    
    ls_req = "ROLLBACK TRANSACTION SIFYB"
    If SQLSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
        li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
        RollbackTran = OK
    End If
    Exit Function
    
suite:
    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
    StdError
End Function

Function SQLSubmit(ByVal ll_Environment As Long, ByVal ll_Database As Long, ByRef ll_Statement As Long, ByVal ls_Request As String, Optional lb_SubmitServer As Boolean = KO, Optional li_FetchMethod = SINGLE_FETCH, Optional li_RowSetSize) As Boolean
'------------------------------------------------------------------
' Name : SQLSubmit
'
' Purpose : Submit a request to the server but verify if we use a
'           cache or not
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'       ll_Statement        Statement identifier
'       ls_Req              Request string to pass to the server
'       lb_SubmitServer     Indicates if we can use server Cursor
'       li_FetchMethod      Indicates if we uses Extended Fetch
'                           method (For Fetch module only)
'       li_RowSetSize       Indicates the row number to read for
'                           the extended fetch method (for Fetch
'                           module only)
'
' review : Jan/13/2000 by AD
'------------------------------------------------------------------
Dim lb_return As Boolean
Dim lb_ClassicLoad As Boolean
Dim ls_StoredProcedureName As String
Dim lb_CreateFile As Boolean
Dim ls_ServerDate As String
Dim ls_ServerTime As String
Dim ls_LocalDate As String
Dim ls_LocalTime As String
Dim ls_Filename As String
Dim ls_drive As String
Dim ls_path As String
Dim li_Status As Integer
Dim ls_req As String

    lb_ClassicLoad = KO
    ls_Filename = ""
    mb_UseCache = KO
    
    If mb_CacheODBC = OK Then
    
        If mb_FirstRequest = OK Then
            mb_FirstRequest = KO
            'Load the list of the cacheable stored procedure
            LoadStoredProcedureList ll_Environment, ll_Database
            lb_ClassicLoad = KO
            ls_Filename = ""
            mb_UseCache = KO
        End If
    
        ls_StoredProcedureName = ExtractStoredProcedure(ls_Request)
    
        'Return false if the name is Empty
        If ls_StoredProcedureName = "CACHE_PROCEDURE_LST" Or SearchStoredProcedureList(ls_StoredProcedureName) = OK Then

            'Have'nt we a cached file ?
            If SearchStoredProcedureCache(ls_StoredProcedureName, ls_Request, ls_Filename, ls_LocalDate, ls_LocalTime) = OK Then
            
                'We find the current date and time of the stored procedure
                ls_req = "EXEC CACHE_PROCEDURE_SEL2 '" & ls_StoredProcedureName & "','" & FormatD(ls_LocalDate, "mm/dd/yyyy") & " " & ls_LocalTime & "'"
                
                If SQLClassicSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
                    Do While li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO
                        li_Status = SQLFetch(ll_Statement)
                        If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
                            ls_ServerDate = ODBCData(ll_Statement, 1)
                            ls_ServerTime = ODBCData(ll_Statement, 2)
                        End If
                    Loop
                    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
                End If
                
                Select Case CompareDateTime(ls_ServerDate, ls_ServerTime, ls_LocalDate, ls_LocalTime)
                    Case -2
                        lb_ClassicLoad = OK
                    Case -1
                        lb_CreateFile = OK
                    Case 0, 1
                        lb_CreateFile = KO
                End Select

            Else
                                
                'We find the current date and time of the stored procedure
                ls_req = "EXEC CACHE_PROCEDURE_SEL2 '" & ls_StoredProcedureName & "',''"
                
                If SQLClassicSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
                    Do While li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO
                        li_Status = SQLFetch(ll_Statement)
                        If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
                            ls_ServerDate = ODBCData(ll_Statement, 1)
                            ls_ServerTime = ODBCData(ll_Statement, 2)
                        End If
                    Loop
                    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
                End If
                                
                'We must create the file
                lb_CreateFile = OK
            End If
            
            If lb_ClassicLoad = KO Then
                Select Case lb_CreateFile
                    Case OK
                        'We must create the file
                        
                        If ls_Filename = "" Then
                            'Choose a new filename
                            GetTempName ms_Directory & "\" & CACHE_DIRECTORY, Mid(ls_StoredProcedureName, 1, 1), ls_drive, ls_path, ls_Filename
                        End If
                        
                        'Create the file
                        lb_return = CreateCache(ls_Filename, ll_Environment, ll_Database, ls_Request)
                        
                        If lb_return = KO Then
                            'We delete the request for the list of existing cached datas
                            DeleteStoredProcedureCache ls_StoredProcedureName, ls_Request
                            lb_ClassicLoad = OK
                        Else
                            'Update or add the information to the list
                            UpdateStoredProcedureCache ls_StoredProcedureName, ls_Request, ls_Filename, ls_ServerDate, ls_ServerTime
                            
                            'We open the file
                            lb_return = OpenCache(ls_Filename)
                            
                            If lb_return = KO Then
                                'We delete the request for the list of existing cached datas
                                DeleteStoredProcedureCache ls_StoredProcedureName, ls_Request
                                lb_ClassicLoad = OK
                            End If
                            
                        End If
                    Case KO
                        'We open the file
                        lb_return = OpenCache(ls_Filename)
                        
                        If lb_return = KO Then
                        
                            'We recreate the file
                            lb_return = CreateCache(ls_Filename, ll_Environment, ll_Database, ls_Request)
                        
                            If lb_return = KO Then
                                'We delete the request for the list of existing cached datas
                                DeleteStoredProcedureCache ls_StoredProcedureName, ls_Request
                                lb_ClassicLoad = OK
                            End If
                        End If
                End Select
            End If
        Else
            lb_ClassicLoad = OK
        End If
    Else
        lb_ClassicLoad = OK
    End If
    
    If lb_ClassicLoad = OK Then
        mb_UseCache = KO
        SQLSubmit = SQLClassicSubmit(ll_Environment, ll_Database, ll_Statement, ls_Request, lb_SubmitServer, li_FetchMethod, li_RowSetSize)
    Else
        mb_UseCache = OK
        SQLSubmit = OK
    End If
    
End Function

Public Function SQLClassicSubmit(ByVal ll_Environment As Long, ByVal ll_Database As Long, ByRef ll_Statement As Long, ByVal ls_req As String, Optional lb_SubmitServer As Boolean = KO, Optional li_FetchMethod = SINGLE_FETCH, Optional li_RowSetSize) As Boolean
'------------------------------------------------------------------
' Name : SQLClassicSubmit
'
' Purpose : Submit a request to the server
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'       ll_Statement        Statement identifier
'       ls_Req              Request string to pass to the server
'       lb_SubmitServer     Indicates if we can use server Cursor
'       li_FetchMethod      Indicates if we uses Extended Fetch
'                           method (For Fetch module only)
'       li_RowSetSize       Indicates the row number to read for
'                           the extended fetch method (for Fetch
'                           module only)
'
' review : Oct/29/1999 by AD
' review : May/31/2000 by JJB print the request that cause the error in the debug immediate mode
'------------------------------------------------------------------
Dim li_Status As Integer
Dim li_Count As Integer
Dim ls_Req2 As String
Dim lb_return As Boolean
Dim ll_lngrows As Long
Dim ll_Statement1 As Long
Dim li_NbColumn As Integer
Dim i As Integer

    SQLClassicSubmit = KO
    
    li_Status = SQLAllocStmt(ll_Database, ll_Statement1)
    
    If li_FetchMethod = MULTIROW_FETCH Then
        'we make a query with the extended fetch method
        li_Status = li_Status And SQLSetStmtOption(ByVal ll_Statement1, SQL_ROWSET_SIZE, li_RowSetSize)
        li_Status = li_Status And SQLSetStmtOption(ByVal ll_Statement1, SQL_CONCURRENCY, SQL_CONCUR_READ_ONLY)
    End If
    
    'we make a query with a server cursor
    If lb_SubmitServer = OK Or li_FetchMethod = MULTIROW_FETCH Then
        li_Status = SQLSetStmtOption(ll_Statement1, SQL_CURSOR_TYPE, SQL_CURSOR_FORWARD_ONLY)
    End If
    
    If li_Status = SQL_SUCCESS Then
        'Verify if we trace the user
        If gs_TraceUser = "Y" Then
'            ls_Req2 = "INSERT INTO Trace_User VALUES ('" _
'                & QuoteParam(ms_UserODBC) & "', GETDATE (), 'No', '" _
'                & QuoteParam(ls_Req) & "')"
            ls_Req2 = "INSERT INTO Trace_User VALUES ('" _
                & QuoteParam(prg.LoginName) & "', GETDATE (), 'No', '" _
                & QuoteParam(ls_req) & "')"
            li_Status = SQLExecDirect(ll_Statement1, ls_Req2, Len(ls_Req2))
            
        End If
        'Execute the query
        li_Status = SQLExecDirect(ll_Statement1, ls_req, Len(ls_req))
        If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
            SQLClassicSubmit = OK
            ll_Statement = ll_Statement1
            
            ' Read the columns number of the query
            SQLNumResultCols ll_Statement, li_NbColumn
            
            ' Resize the description buffer
            ReDim met_DescColumns(li_NbColumn)

            For i = 1 To li_NbColumn
                ' Read the description of the current column
                SQLDescribeCol ll_Statement, i, met_DescColumns(i).ColumnName, 20, met_DescColumns(i).SizeColName, met_DescColumns(i).DataType, met_DescColumns(i).SizeColumn, met_DescColumns(i).Scale, met_DescColumns(i).Nullable
            Next
            
        Else
            'The query fails.
            'If it's the Extended Fetch method, we don't get the error because the Fetch module make a new
            ' trial with the standard method
            If li_FetchMethod <> MULTIROW_FETCH Then
                'We retrieves all the error
                ErrorGet ll_Environment, ll_Database, ll_Statement1
                'We close the current statement
                li_Status = SQLFreeStatement(ll_Statement1, SQL_DROP)
                
                'We write the error to the database
                For li_Count = 1 To gi_ErrorCount
                    ls_Req2 = "EXEC Error_Log_Insert '" _
                        & App.EXEName & "', '" _
                        & App.Major & "." & App.Minor & "." _
                        & App.Revision & "', '" _
                        & QuoteParam(ls_req) & "', '" _
                        & QuoteParam(gs_NativeError(li_Count)) & "', '" _
                        & QuoteParam(gs_ErrorMesssage(li_Count)) & "'"
                    If mb_FirstTest = False Then
                        'debug.print "Error SQL logged :"
                        'debug.print ls_Req
                        'debug.print "(" & gs_NativeError(li_Count) & ") - " & QuoteParam(gs_ErrorMesssage(li_Count))
                        lb_return = SQLSubmitError(ll_Database, ls_Req2)
                    Else
                        If LCase(Trim(gs_ErrorMesssage(li_Count))) = LCase(Trim("[Microsoft][ODBC SQL Server Driver]Connection is busy with results for another hstmt")) Then
                            mb_LockedStatement = True
                        Else
                            'debug.print "Error SQL logged :"
                            'debug.print ls_Req
                            'debug.print "(" & gs_NativeError(li_Count) & ") - " & QuoteParam(gs_ErrorMesssage(li_Count))
                        End If
                    End If
                    If gb_NoMessage = False Then
                        Select Case gs_NativeError(li_Count)
                        Case 547: SendMessage 97, "The data you tried to delete is still referenced by another entity.", "E", , , mb_NCS_Method
                        Case 2601, 2626, 2627: SendMessage 59, "  This data already exists", "E", , , mb_NCS_Method
                        End Select
                    End If
                Next li_Count
                'Verify if we trace the user
                If gs_TraceUser = "Y" And mb_FirstTest = False Then
                    'We open a new statement to write the trace user
                    li_Status = SQLAllocStmt(ll_Database, ll_Statement1)
'                    ls_Req2 = "UPDATE Trace_User SET TU_error = 'Yes' " _
'                        & "WHERE TU_user = '" & ms_UserODBC _
'                        & "' AND TU_date = (SELECT MAX(TU_date) FROM Trace_User " _
'                            & "WHERE TU_user = '" & ms_UserODBC & "')"
                    ls_Req2 = "UPDATE Trace_User SET TU_error = 'Yes' " _
                        & "WHERE TU_user = '" & prg.LoginName _
                        & "' AND TU_date = (SELECT MAX(TU_date) FROM Trace_User " _
                            & "WHERE TU_user = '" & prg.LoginName & "')"
                    li_Status = SQLExecDirect(ll_Statement1, ls_Req2, Len(ls_Req2))
                    'We close the current statement
                    li_Status = SQLFreeStatement(ll_Statement1, SQL_DROP)
                End If
            Else
                'We close the current statement
                li_Status = SQLFreeStatement(ll_Statement1, SQL_DROP)
            End If
        End If
    Else
        'We retrieves all the error
        ErrorGet ll_Environment, ll_Database, ll_Statement1
        
        If LCase(Trim(gs_ErrorMesssage(1))) = LCase(Trim("[Microsoft][ODBC SQL Server Driver]Connection is busy with results for another hstmt")) Then
            mb_LockedStatement = True
        End If
    End If

End Function

Public Function SQLSubmitServer(ByVal ll_Environment As Long, ByVal ll_Database As Long, ByRef ll_Statement As Long, ByVal ls_req As String) As Boolean
'------------------------------------------------------------------
' Name : SQLSubmitServer
'
' Purpose : Submit a request to the server with server cursor
'           (For compatibility, must use SQLSubmit now)
'
' Parameters :
'       ll_Environment      Environement identifier
'       ll_Database         Database identifier
'       ll_Statement        Statement identifier
'       ls_Req              Request string to pass to the server
'       li_FetchMethod      Indicates if we uses Extended Fetch
'                           method (For Fetch module only)
'       li_RowSetSize       Indicates the row number to read for
'                           the extended fetch method (for Fetch
'                           module only)
'
' review : 09/17/1999 by AD
'------------------------------------------------------------------
   
    SQLSubmitServer = SQLSubmit(ll_Environment, ll_Database, ll_Statement, ls_req, OK)

End Function

Function SQLSubmitError(ByVal ll_Database As Long, ByVal ls_req As String) As Boolean
'------------------------------------------------------------------
' Name : SQLSubmitError
'
' Purpose : Write an error to the database
'
' Parameters :
'       ll_Database         Database identifier
'       ls_Req              Request string to pass to the server
'
' review : Sep/17/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ll_Statement As Long

    SQLSubmitError = KO
    
    On Error GoTo SQLSubmitError_Err
    
    'Create a new statement
    li_Status = SQLAllocStmt(ll_Database, ll_Statement)
    
    If li_Status = SQL_SUCCESS Then
        'Write the error
        li_Status = SQLExecDirect(ll_Statement, ls_req, Len(ls_req))
        If li_Status = SQL_SUCCESS Or _
            li_Status = SQL_SUCCESS_WITH_INFO Then
                SQLSubmitError = OK
        End If
    End If
    
    'Delete the statement
    SQLFreeStmt ll_Statement, SQL_DROP
    
    Exit Function

SQLSubmitError_Err:
    SQLFreeStmt ll_Statement, SQL_DROP
End Function

Function SQLFreeStatement(ByRef ll_Statement As Long, ByVal ll_type As Integer) As Integer
'------------------------------------------------------------------
' Name : SQLFreeStatement
'
' Purpose : Call SQLFreeStmt if necessary or close the file in case of cache
'
' Parameters :
'       ll_Statement        Statement identifier
'       ll_Type             Indicates the type of free (SQL_DROP for example)
'
' review : Oct/29/1999 by AD
'------------------------------------------------------------------
Dim li_Status As Integer
Dim ll_Statement1 As Long
Dim lb_Result As BinaryFile_Error

    On Error Resume Next

    If mb_UseCache = OK Then
        lb_Result = mo_BinaryFile.CloseFile
        If lb_Result <> BF_OK Then
            SQLFreeStatement = SQL_ERROR
        Else
            SQLFreeStatement = SQL_SUCCESS
        End If
    Else
        li_Status = 0
        ll_Statement1 = ll_Statement
        If ll_Statement <> 0 Then
            li_Status = SQLFreeStmt(ll_Statement1, ll_type)
            ll_Statement = 0
            SQLFreeStatement = li_Status
        End If
    End If
    
End Function

Private Function CreateCache(ls_Filename As String, ByVal ll_Environment As Long, ByVal ll_Database As Long, ByVal ls_req As String) As Boolean
'------------------------------------------------------------------
' Name : CreateCache
'
' Purpose : Create the cached file
'
' Parameters :
'       ls_FileName             The file containing the cached
'                               data
'
' Return :
'       OK if the file is created otherwise KO
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_Directory As String
Dim lo_Return As BinaryFile_Error
Dim ll_Statement As Long
Dim li_ColumnNumber As Integer
Dim lb_Single() As Boolean
Dim lb_Text() As Boolean
Dim la_data() As String
Dim i As Integer
Dim ls_ColumnName As String
Dim li_SizeColName As Integer
Dim li_dataType As Integer
Dim ll_sizeColumn As Long
Dim li_Scale As Integer
Dim li_Nullable As Integer
Dim li_Status As Integer

    ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
    
    On Error GoTo CreateCache_Err
    
    CreateCache = KO
    
    'Create the submit
    If SQLClassicSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
    
        ' Read the columns number of the query
        SQLNumResultCols ll_Statement, li_ColumnNumber
        
        ReDim lb_Single(li_ColumnNumber)
        ReDim la_data(li_ColumnNumber)
        ReDim lb_Text(li_ColumnNumber)
        
        'Read the type of each column
        For i = 1 To li_ColumnNumber
            ' Read the description of the current column
            SQLDescribeCol ll_Statement, i, ls_ColumnName, 20, li_SizeColName, li_dataType, ll_sizeColumn, li_Scale, li_Nullable
            
            If li_dataType = SQL_REAL Then
                lb_Single(i) = OK
            Else
                lb_Single(i) = KO
            End If
            
            If ll_sizeColumn > 200 Then
                lb_Text(i) = OK
            Else
                lb_Text(i) = KO
            End If
        Next
            
        'Create the file
        mo_BinaryFile.ColumnNumber = li_ColumnNumber
        mo_BinaryFile.NameFile = ls_Directory & ls_Filename
        lo_Return = mo_BinaryFile.CreateFile()
        
        If lo_Return <> BF_OK Then
            li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
            Exit Function
        End If
        
        Do While li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO
            li_Status = SQLFetch(ll_Statement)
            If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
                
                'Read the data (in function of type)
                For i = 1 To li_ColumnNumber
                    If lb_Single(i) = KO Then
                        If lb_Text(i) = KO Then
                            la_data(i - 1) = ODBCData(ll_Statement, i)
                        Else
                            la_data(i - 1) = ODBCMemoGet(ll_Statement, i)
                        End If
                    Else
                        la_data(i - 1) = ODBCDataSingle(ll_Statement, i)
                    End If
                Next
                mo_BinaryFile.WriteLine la_data
                
            End If
        Loop
        
        'Close the file
        mo_BinaryFile.CloseFile
        
        li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
        
        CreateCache = OK
        
    End If
        
    Exit Function

CreateCache_Err:
    li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
        
End Function

Private Function OpenCache(ls_Filename As String) As Boolean
'------------------------------------------------------------------
' Name : OpenCache
'
' Purpose : Open the cached file
'
' Parameters :
'       ls_FileName             The file containing the cached
'                               data
'
' Return :
'       OK if the file is opened otherwise KO
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_Directory As String
Dim lo_Return As BinaryFile_Error

    ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
    
    mo_BinaryFile.NameFile = ls_Directory & ls_Filename
    lo_Return = mo_BinaryFile.OpenFile()
    
    ReDim ms_Data(mo_BinaryFile.ColumnNumber)
    
    If lo_Return <> BF_OK Then
        OpenCache = KO
    Else
        OpenCache = OK
    End If
    
End Function

Private Function SearchStoredProcedureCache(ls_StoredProcedure As String, ls_Request As String, ls_Filename As String, ls_ServerDate As String, ls_ServerTime As String) As Boolean
'------------------------------------------------------------------
' Name : SearchStoredProcedureCache
'
' Purpose : Search the cache to the list of the cached stored
'           procedure
'
' Parameters :
'       ls_StoredProcedure      the name of the stored procedure
'                               cached
'       ls_Request              The request name cached
'       ls_FileName             The file containing the cached
'                               data
'       ls_ServerDate           Date of the last cache
'       ls_ServerTime           Time of the last cache
'
' Return :
'       OK if the request is found otherwise KO
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_FirstLetter As String
Dim lo_File1 As New BinaryFile
Dim la_data(4) As String
Dim ls_Directory As String
Dim ls_GoodFile As String

    ls_FirstLetter = Mid(ls_StoredProcedure, 1, 1)
    
    ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
    ls_GoodFile = ls_Directory & ls_FirstLetter & "." & CACHE_EXTENSION
    
    SearchStoredProcedureCache = KO

    'Verify if the cache list for this letter exist
    If Dir(ls_GoodFile, vbNormal) <> "" Then
        
        lo_File1.NameFile = ls_GoodFile
        
        lo_File1.OpenFile
        Do While Not lo_File1.isEOF
            lo_File1.ReadLine la_data
            If la_data(0) = ls_Request Then
                ls_Filename = la_data(1)
                ls_ServerDate = la_data(2)
                ls_ServerTime = la_data(3)
                lo_File1.CloseFile
                SearchStoredProcedureCache = OK
                lo_File1.CloseFile
                Exit Function
            End If
        Loop
        
        lo_File1.CloseFile
    End If
    
End Function

Private Sub DeleteStoredProcedureCache(ls_StoredProcedure As String, ls_Request As String)
'------------------------------------------------------------------
' Name : DeleteStoredProcedureCache
'
' Purpose : Delete the actual cache to the list of the cached
'           stored procedure
'
' Parameters :
'       ls_StoredProcedure      the name of the stored procedure
'                               cached
'       ls_Request              The request name cached
'
' Return : Nothing
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_FirstLetter As String
Dim ls_drive As String
Dim ls_path As String
Dim ls_TempName As String
Dim lo_File1 As New BinaryFile
Dim lo_File2 As New BinaryFile
Dim la_data(4) As String
Dim ls_Directory As String
Dim ls_GoodFile As String
Dim lb_return As BinaryFile_Error

    lo_File1.ForceClose = OK
    lo_File1.OverWrite = OK
    lo_File2.ForceClose = OK
    lo_File2.OverWrite = OK
    
    ls_FirstLetter = Mid(ls_StoredProcedure, 1, 1)
    
    ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
    ls_GoodFile = ls_Directory & ls_FirstLetter & "." & CACHE_EXTENSION
    
    GetTempName ls_Directory, ls_FirstLetter, ls_drive, ls_path, ls_TempName
    
    ls_TempName = ls_Directory & ls_TempName
    lo_File1.ColumnNumber = 4
    lo_File1.NameFile = ls_TempName
    lo_File2.NameFile = ls_GoodFile
    
    On Error GoTo DeleteStoredProcedureCache_Err
    lo_File1.CreateFile
    lb_return = lo_File2.OpenFile
    If lb_return = BF_OK Then
        Do While Not lo_File2.isEOF
            lo_File2.ReadLine la_data
            If Not lo_File2.isEOF Then
                If la_data(0) <> ls_Request Then lo_File1.WriteLine la_data
            End If
        Loop
    End If
    lo_File2.CloseFile
    lo_File1.CloseFile
    
    On Error Resume Next
    Kill ls_GoodFile
    FileCopy ls_TempName, ls_GoodFile
    Kill ls_TempName
    
    Exit Sub
    
DeleteStoredProcedureCache_Err:
    lo_File1.CloseFile
    lo_File2.CloseFile
End Sub

Private Sub UpdateStoredProcedureCache(ls_StoredProcedure As String, ls_Request As String, ls_Filename As String, ls_ServerDate As String, ls_ServerTime As String)
'------------------------------------------------------------------
' Name : UpdateStoredProcedureCache
'
' Purpose : Update the cache to the list of the cached stored
'           procedure
'
' Parameters :
'       ls_StoredProcedure      the name of the stored procedure
'                               cached
'       ls_Request              The request name cached
'       ls_FileName             The file containing the cached
'                               data
'       ls_ServerDate           Date of the last cache
'       ls_ServerTime           Time of the last cache
'
' Return : Nothing
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_FirstLetter As String
Dim ls_drive As String
Dim ls_path As String
Dim ls_TempName As String
Dim lo_File1 As New BinaryFile
Dim lo_File2 As New BinaryFile
Dim la_data(4) As String
Dim ls_Directory As String
Dim ls_GoodFile As String
Dim lo_Return As BinaryFile_Error
Dim lb_Found As Boolean

    lo_File1.ForceClose = OK
    lo_File1.OverWrite = OK
    lo_File2.ForceClose = OK
    lo_File2.OverWrite = OK
    
    ls_FirstLetter = Mid(ls_StoredProcedure, 1, 1)
    
    ls_Directory = ms_Directory & "\" & CACHE_DIRECTORY
    ls_GoodFile = ls_Directory & ls_FirstLetter & "." & CACHE_EXTENSION
    
    GetTempName ls_Directory, ls_FirstLetter, ls_drive, ls_path, ls_TempName
    
    ls_TempName = ls_Directory & ls_TempName
    lo_File1.ColumnNumber = 4
    lo_File1.NameFile = ls_TempName
    lo_File2.NameFile = ls_GoodFile
    
    On Error GoTo UpdateStoredProcedureCache_Err
    lo_File1.CreateFile
    lo_Return = lo_File2.OpenFile
    lb_Found = KO
    If lo_Return = BF_OK Then
        Do While Not lo_File2.isEOF
            lo_File2.ReadLine la_data
            If Not lo_File2.isEOF Then
                If la_data(0) = ls_Request Then
                    'We change the informations of the cached request
                    lb_Found = OK
                    la_data(2) = ls_ServerDate
                    la_data(3) = ls_ServerTime
                End If
                lo_File1.WriteLine la_data
            End If
        Loop
        lo_File2.CloseFile
    End If
    
    If lb_Found = KO Then
        la_data(0) = ls_Request
        la_data(1) = ls_Filename
        la_data(2) = ls_ServerDate
        la_data(3) = ls_ServerTime
        lo_File1.WriteLine la_data
    End If
    
    lo_File1.CloseFile
    
    On Error Resume Next
    Kill ls_GoodFile
    FileCopy ls_TempName, ls_GoodFile
    Kill ls_TempName
        
    Exit Sub

UpdateStoredProcedureCache_Err:
    lo_File1.CloseFile
    lo_File2.CloseFile
    
    Set lo_File1 = Nothing
    Set lo_File2 = Nothing
End Sub

Private Function ExtractStoredProcedure(ByVal ls_req As String) As String
'------------------------------------------------------------------
' Name : ExtractStoredProcedure
'
' Purpose : Extract from the request the name of the stored
'           procedure
'
' Parameters :
'       ls_Req              Request string to pass to the server
'
' Return :
'       The name of the stored procedure or "" if nothing
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim li_pos As Integer
Dim li_Pos2 As Integer

    'In first, search 'Exec'
    li_pos = InStr(1, ls_req, "exec", vbTextCompare)
    If li_pos > 0 Then
        'We have find Exec, so the next name is the stored procedure
        li_pos = li_pos + 5
        li_Pos2 = InStr(li_pos, ls_req, " ", vbTextCompare)
        If li_Pos2 = 0 Then li_Pos2 = Len(ls_req) + 1
        ExtractStoredProcedure = Mid(ls_req, li_pos, li_Pos2 - li_pos)
        Exit Function
    End If
    
    'We hav'nt the 'Exec' function
    'We try to find 'select', 'insert' or 'delete' procedure
    li_pos = InStr(1, ls_req, "insert", vbTextCompare)
    li_pos = li_pos + InStr(1, ls_req, "select", vbTextCompare)
    li_pos = li_pos + InStr(1, ls_req, "delete", vbTextCompare)
    If li_pos > 0 Then
        'We don't execute stored procedure
        ExtractStoredProcedure = ""
        Exit Function
    End If
    
    'We execute a stored procedure but without 'Exec' function, so the first name is the stored procedure
    li_Pos2 = InStr(1, LTrim(ls_req), " ", vbTextCompare)
    If li_Pos2 = 0 Then li_Pos2 = Len(ls_req) + 1
    ExtractStoredProcedure = Mid(LTrim(ls_req), 1, li_Pos2 - 1)
    
End Function

Private Sub LoadStoredProcedureList(ByVal ll_Environment As Long, ByVal ll_Database As Long)
'------------------------------------------------------------------
' Name : LoadStoredProcedureList
'
' Purpose : Extract from the request the name of the stored
'           procedure
'
' Parameters : Nothing
'
' Return : Nothing
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim i As Integer
Dim ls_req As String
Dim li_Status As Integer
Dim ls_Value As String
Dim ll_Statement As Long
    
    '===========================================================================================
    ls_req = "EXEC CACHE_PROCEDURE_LST"
    
    If SQLSubmit(ll_Environment, ll_Database, ll_Statement, ls_req) Then
        Do While li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO
            li_Status = SQLFetch(ll_Statement)
            If li_Status = SQL_SUCCESS Or li_Status = SQL_SUCCESS_WITH_INFO Then
                ls_Value = ODBCData(ll_Statement, 1)
                mo_StoredProcedureList.Add ls_Value, ls_Value
            End If
        Loop
        li_Status = SQLFreeStatement(ll_Statement, SQL_DROP)
    End If

    mb_FirstRequest = KO

End Sub

Private Function SearchStoredProcedureList(ls_StoredProcedure As String) As Boolean
'------------------------------------------------------------------
' Name : SearchStoredProcedureList
'
' Purpose : Indicates if the stored procedure is in the list
'
' Parameters :
'       ls_StoredProcedure      the name of the stored procedure to
'                               find
'
' Return :
'       OK if the stored procedure is in the list otherwise KO
'
' review : Jan/17/2000 by AD
'------------------------------------------------------------------
Dim ls_Value As String

    SearchStoredProcedureList = KO
    
    On Error GoTo SearchStoredProcedureList_Err
    
    ls_Value = mo_StoredProcedureList(ls_StoredProcedure)
    
    SearchStoredProcedureList = OK
        
    Exit Function

SearchStoredProcedureList_Err:

End Function

Public Function NCS_Connection(ByVal ls_DSN As String, ByVal ls_DatabaseName As String, ByVal ls_AppIdentifier As String, ByVal ls_Login As String, ByVal ls_Password As String, Optional ByVal ls_OtherParameters As String) As Boolean
Dim ll_Environment As Long
Dim ll_Database As Long

    NCS_Disconnection

    ms_ConnectionString = ODBCConnectBuild(ls_DSN, ls_DatabaseName, ls_AppIdentifier, ls_Login, ls_Password, ls_OtherParameters)
    If ODBCInit(ll_Environment, ll_Database) = False Then
        NCS_Connection = False
        Exit Function
    End If
    If ODBCDataOpen(ll_Database, ms_ConnectionString) = True Then
        mi_ConnectionNumber = 1
        mi_StatementNumber = 0
        ReDim met_Connection(mi_ConnectionNumber)
        met_Connection(mi_ConnectionNumber).Connection = mi_ConnectionNumber
        met_Connection(mi_ConnectionNumber).Environment = ll_Environment
        met_Connection(mi_ConnectionNumber).Database = ll_Database
        ml_CurrentEnvironment = ll_Environment
        ml_CurrentDatabase = ll_Database
        mi_CurrentConnection = mi_ConnectionNumber
        mb_NCS_Method = True
        NCS_Connection = True
    Else
'        ODBCDataClose ll_Environment, ll_Database
        NCS_Connection = False
    End If
        
End Function

Public Sub NCS_Disconnection()
Dim i As Integer
Dim li_Connection As Integer
Dim ll_Statement As Long

    For i = mi_StatementNumber To 1 Step -1
        'Use a variable because if not, we cannot resize the tab
        ll_Statement = met_Statement(i).Statement
        NCS_FreeStatement ll_Statement, SQL_DROP
    Next

    For i = mi_ConnectionNumber To 1 Step -1
        'Use a variable because if not, we cannot resize the tab
        li_Connection = met_Connection(i).Connection
        NCS_DeleteConnection li_Connection, True
    Next
    ReDim met_Connection(0)
    ReDim met_Statement(0)
    mi_ConnectionNumber = 0
    mi_StatementNumber = 0
    mi_CurrentConnection = 0
    ml_CurrentDatabase = 0
    ml_CurrentEnvironment = 0
End Sub

Private Function NCS_NewConnection() As Boolean
'Verify if we have a place in the tab
Dim li_Position As Integer
Dim li_Max As Integer
Dim i As Integer
Dim ll_Environment As Long
Dim ll_Database As Long

    li_Position = 0
    For i = 1 To mi_ConnectionNumber
        If met_Connection(i).Connection = 0 Then
            li_Position = i
            Exit For
        End If
    Next

    li_Max = 0
    For i = 1 To mi_ConnectionNumber
        If met_Connection(i).Connection > li_Max Then
            li_Max = met_Connection(i).Connection
        End If
    Next
    li_Max = li_Max + 1

    If li_Position = 0 Then
        mi_ConnectionNumber = mi_ConnectionNumber + 1
        ReDim Preserve met_Connection(mi_ConnectionNumber)
        li_Position = mi_ConnectionNumber
    End If

    If ODBCInit(ll_Environment, ll_Database) = False Then
        NCS_NewConnection = False
        Exit Function
    End If
    If ODBCDataOpen(ll_Database, ms_ConnectionString) = True Then
        met_Connection(li_Position).Connection = li_Max
        met_Connection(li_Position).Environment = ll_Environment
        met_Connection(li_Position).Database = ll_Database
        ml_CurrentEnvironment = ll_Environment
        ml_CurrentDatabase = ll_Database
        mi_CurrentConnection = li_Position
        NCS_NewConnection = True
    Else
        ODBCDataClose ll_Environment, ll_Database
        NCS_NewConnection = False
    End If

End Function

Private Sub NCS_DeleteConnection(li_Connection As Integer, Optional lb_Last As Boolean = False)
'If it's not the last place, we don't resize the tab
Dim i, j As Integer
Dim li_ConnectionNumber As Integer
Dim li_UseConnection As Integer

    li_ConnectionNumber = 0
    For i = 1 To mi_ConnectionNumber
        If met_Connection(i).Connection <> 0 Then
            li_ConnectionNumber = li_ConnectionNumber + 1
        End If
    Next
    
    For i = 1 To mi_ConnectionNumber
        If met_Connection(i).Connection = li_Connection And (li_ConnectionNumber > 1 Or lb_Last = True) Then
            ODBCDataClose met_Connection(i).Environment, met_Connection(i).Database
            If i = mi_ConnectionNumber And (li_ConnectionNumber <> 1 Or lb_Last = True) Then
                mi_ConnectionNumber = mi_ConnectionNumber - 1
                ReDim Preserve met_Connection(mi_ConnectionNumber)
                ml_CurrentDatabase = met_Connection(mi_ConnectionNumber).Database
                mi_CurrentConnection = met_Connection(mi_ConnectionNumber).Connection
                ml_CurrentEnvironment = met_Connection(mi_ConnectionNumber).Environment
            Else
                met_Connection(i).Connection = 0
                met_Connection(i).Database = 0
                met_Connection(i).Environment = 0
                li_UseConnection = 0
                For j = 1 To mi_ConnectionNumber
                    If met_Connection(mi_ConnectionNumber).Connection <> 0 Then
                        li_UseConnection = j
                        Exit For
                    End If
                Next
                ml_CurrentDatabase = met_Connection(j).Database
                mi_CurrentConnection = met_Connection(j).Connection
                ml_CurrentEnvironment = met_Connection(j).Environment
            End If
            'Exit For
        End If
    Next

End Sub

Private Sub NCS_AddStatement(ll_Statement As Long)
'Verify if we have a place in the tab
Dim i As Integer
Dim li_Position As Integer

    li_Position = 0

    For i = 1 To mi_StatementNumber
        If met_Statement(i).Statement = 0 Then
            li_Position = i
            Exit For
        End If
    Next

    If li_Position <> 0 Then
        met_Statement(li_Position).Statement = ll_Statement
        met_Statement(li_Position).Connection = mi_CurrentConnection
        Exit Sub
    End If

    mi_StatementNumber = mi_StatementNumber + 1
    ReDim Preserve met_Statement(mi_StatementNumber)

    met_Statement(mi_StatementNumber).Statement = ll_Statement
    met_Statement(mi_StatementNumber).Connection = mi_CurrentConnection

End Sub

Private Sub NCS_DeleteStatement(ll_Statement As Long)
'If all statement of one connection are deleted, we delete the connection
'If it's not the last place, we don't resize the tab
Dim i As Integer
Dim li_Connection As Integer
Dim li_Position As Integer

    'Delete statement
    For i = 1 To mi_StatementNumber
        If met_Statement(i).Statement = ll_Statement Then
            li_Connection = met_Statement(i).Connection
            If i = mi_StatementNumber Then
                mi_StatementNumber = mi_StatementNumber - 1
                ReDim Preserve met_Statement(mi_StatementNumber)
            Else
                met_Statement(i).Connection = 0
                met_Statement(i).Statement = 0
            End If
            Exit For
        End If
    Next

    'Find other connection
    li_Position = 0
    For i = 1 To mi_StatementNumber
        If met_Statement(i).Connection = li_Connection Then
            li_Position = i
            Exit For
        End If
    Next
    If li_Position = 0 Then
        NCS_DeleteConnection li_Connection
    End If
    
End Sub

Public Function NCS_Submit(ByRef ll_Statement As Long, ByVal ls_Request As String, Optional lb_SubmitServer As Boolean = KO, Optional li_FetchMethod = SINGLE_FETCH, Optional li_RowSetSize) As Boolean
Dim lb_Result As Boolean

    mb_FirstTest = True
    mb_LockedStatement = False
    
    lb_Result = SQLSubmit(ml_CurrentEnvironment, ml_CurrentDatabase, ll_Statement, ls_Request, lb_SubmitServer, li_FetchMethod, li_RowSetSize)

    If lb_Result = True Then
        NCS_Submit = True
        'New Statement
        NCS_AddStatement ll_Statement
        Exit Function
    End If
    
    mb_FirstTest = False
    
    If mb_LockedStatement = False Then Exit Function
    
    NCS_NewConnection
    lb_Result = SQLSubmit(ml_CurrentEnvironment, ml_CurrentDatabase, ll_Statement, ls_Request, lb_SubmitServer, li_FetchMethod, li_RowSetSize)
    If lb_Result = True Then
        NCS_Submit = True
        'New Statement
        NCS_AddStatement ll_Statement
    End If
    
End Function

Public Function NCS_ClassicSubmit(ByRef ll_Statement As Long, ByVal ls_Request As String, Optional lb_SubmitServer As Boolean = KO, Optional li_FetchMethod = SINGLE_FETCH, Optional li_RowSetSize) As Boolean
Dim lb_Result As Boolean

    mb_FirstTest = True
    mb_LockedStatement = False
    
    lb_Result = SQLClassicSubmit(ml_CurrentEnvironment, ml_CurrentDatabase, ll_Statement, ls_Request, lb_SubmitServer, li_FetchMethod, li_RowSetSize)

    If lb_Result = True Then
        NCS_ClassicSubmit = True
        'New Statement
        NCS_AddStatement ll_Statement
        Exit Function
    End If
    
    mb_FirstTest = False
    
    If mb_LockedStatement = False Then Exit Function
    
    NCS_NewConnection
    lb_Result = SQLClassicSubmit(ml_CurrentEnvironment, ml_CurrentDatabase, ll_Statement, ls_Request, lb_SubmitServer, li_FetchMethod, li_RowSetSize)
    If lb_Result = True Then
        NCS_ClassicSubmit = True
        'New Statement
        NCS_AddStatement ll_Statement
    End If
    
End Function

Public Function NCS_BeginTran() As Boolean
    
    NCS_BeginTran = BeginTran(ml_CurrentEnvironment, ml_CurrentDatabase)
    
End Function

Public Function NCS_RollbackTran() As Boolean
    
    NCS_RollbackTran = RollbackTran(ml_CurrentEnvironment, ml_CurrentDatabase)
    
End Function

Public Function NCS_CommitTran() As Boolean
    
    NCS_CommitTran = CommitTran(ml_CurrentEnvironment, ml_CurrentDatabase)
    
End Function

Function NCS_GetData(ByVal ll_Statement As Long, ByVal li_NbColumn As Integer) As String

    NCS_GetData = ODBCData(ll_Statement, li_NbColumn)

End Function

Function NCS_GetMemo(ByVal ll_Statement As Long, ByVal li_Column As Integer) As String

    NCS_GetMemo = ODBCMemoGet(ll_Statement, li_Column)

End Function

Function NCS_RowCount(ByVal ll_Statement As Long, ByRef ll_rowNumber As Long) As Integer
    NCS_RowCount = SQLRowCount(ll_Statement, ll_rowNumber)
End Function

Function NCS_FreeStatement(ByRef ll_Statement As Long, ByVal ll_type As Integer) As Integer
Dim ll_Statement1 As Long
Dim lb_Found As Boolean
Dim i As Integer

    lb_Found = False
    For i = 1 To mi_StatementNumber
        If met_Statement(i).Statement = ll_Statement Then
            lb_Found = True
            Exit For
        End If
    Next

    If lb_Found = True Then
        ll_Statement1 = ll_Statement
        NCS_FreeStatement = SQLFreeStatement(ll_Statement, ll_type)
    
        NCS_DeleteStatement ll_Statement1
    Else
        ll_Statement = 0
    End If
End Function

Function NCS_Fetch(ByVal ll_Statement As Long) As Integer
    
    NCS_Fetch = SQLFetch(ll_Statement)
    
End Function

Function NCS_NumResultCols(ByVal al_statement As Long) As Integer
Dim li_colNumber As Integer

    SQLNumResultCols al_statement, li_colNumber
    
    NCS_NumResultCols = li_colNumber
    
End Function
